###############################################################################
#                                                                             #
#                      Object Information for Office 365                      #
#                                    V3.97                                    #
#                       (C)2006-2019 KoXo Developpement                       #
#                                                                             #
###############################################################################
function log_message
{
Param($Header,$Message)	
$LogDate=Get-Date
$LogInfo="[$($LogDate)]: [$($Header)] $($Message)"
$LogInfo | Out-File -FilePath $LogFile -Encoding UTF8 -Append -Force
}
###############################################################################
function log_total_errors_count
{
$S="TotalErrorsCount=$($global:total_errors_count)"
$S | Out-File -FilePath $LogFile -Encoding UTF8 -Append -Force
}
###############################################################################
function log_errors_count
{
Param($UserID)
$S="ErrorsCount[$($UserID)]=$($global:errors_count)"
$S | Out-File -FilePath $LogFile -Encoding UTF8 -Append -Force
}
###############################################################################
function check_version
{
$S=Get-Host
if ($S.Version.Major -lt "2")
   {
   $S="Powershell's version must be greater or equal to 2.0 !"
   log_message "Version" $S
   $global:total_errors_count++
   write-host $S -foregroundcolor "red"
   write-host "Press 'Enter' to continue ..."
   Read-Host
   exit
   }
}
###############################################################################
function welcome
{
write-host "Object Information for Office 365 $($Version)" -foregroundcolor "white"
write-host "(C) 2006-2019 KoXo Developpement" -foregroundcolor "white"
}
###############################################################################
function connect
{
Param([string] $MyURL,$MyName,$MyPassword,$ProxyEnabled,$ProxyUser,$ProxyPassword)
$error.Clear()
#Proxy
if ($ProxyEnabled)
   {
   if (![String]::IsNullorEmpty($ProxyUser))
      { 
      $SecuredPassword=ConvertTo-SecureString $ProxyPassword -AsPlainText -Force
      $ProxyCred=New-Object System.Management.Automation.PSCredential $ProxyUser, $SecuredPassword
      $PSSessionOption=New-PSSessionOption -ProxyAccessType IEConfig -ProxyAuthentication Negotiate -ProxyCredential $ProxyCred
      $S="Connection with Internet Explorer proxy parameters, User=$($ProxyUser)"
	  log_message "OK" $S
      }
      else 
      {
      $PSSessionOption=New-PSSessionOption -ProxyAccessType IEConfig -ProxyAuthentication Negotiate
      $S="Connection with Internet Explorer proxy parameters"
	  log_message "OK" $S
      }
   write-host $S -foregroundcolor "cyan"   
   }
# Set credential
if ( ([String]::IsNullorEmpty($MyPassword)) -or ([String]::IsNullorEmpty($MyName))  )
   {
   $Cred=Get-Credential
   }
   else
   {
   $SecuredPassword=ConvertTo-SecureString $MyPassword -AsPlainText -Force
   $Cred=New-Object System.Management.Automation.PSCredential $MyName, $SecuredPassword
   }
# Creating new runspace. 
if ($ProxyEnabled)
   {
   $Script:RS=New-PSSession -ConfigurationName microsoft.exchange -ConnectionUri $MyURL -Credential $Cred -Authentication Basic -AllowRedirection -SessionOption $PSSessionOption -ErrorAction SilentlyContinue
   }
   else
   {
   $Script:RS=New-PSSession -ConfigurationName microsoft.exchange -ConnectionUri $MyURL -Credential $Cred -Authentication Basic -AllowRedirection -ErrorAction SilentlyContinue
   }
if ([String]::IsNullorEmpty($error[0]))
   {
    $S="Runspace Creation was successful = $($MyURL)"
	log_message "OK" $S
	# Import-PSSession $Script:RS
   } 
   else 
   {
   $global:total_errors_count++
   $S="Runspace creation was unsuccessful to $($MyURL) || $($error[0])"
   log_message "ERROR" $S
   # Display error
   write-host $S -foregroundcolor "red"
   write-host "Press 'Enter' to continue ..."
   Read-Host   
   }   
 return $Cred
}
###############################################################################
function close
{
$error.Clear()
$Script:RS | Remove-PSSession -ErrorAction SilentlyContinue
if ([String]::IsNullorEmpty($Error[0]))
	{
    $S="Successfully cleared Runspace"
	log_message "OK" $S
	}
	else
	{
	$global:total_errors_count++
    $S="Clearing Runspace was unsuccessful. Error = $($error[0])"
    log_message "ERROR" $S
	}
}
###############################################################################
## Install MSOnline Cmdlets 
function Install-MSOnlineCmdlets()
{
$Res=$false
$DotNET35Path="HKLM:\SOFTWARE\Microsoft\NET Framework Setup\NDP\v3.5"
$DotNet35Present=((Test-Path $DotNET35Path) -and ((Get-ItemProperty $DotNET35Path).Install -eq "1"))
if ($DotNet35Present)
   {
   # Trying to import module
   if (-not(Get-Module -Name "MSOnline")){Get-Module -ListAvailable | Where-Object {$_.name -eq "MSOnline"} | Import-Module}
   # Module is not present
   if (-not(Get-Module -Name "MSOnline"))
      {
      Write-Host "MSOnline Powershell module is not present !" -Foregroundcolor "Red"
      $wc=New-Object System.Net.WebClient
      # Sign-in Assistant
      $SigninAssistantPath="HKLM:\SOFTWARE\Microsoft\MSOIdentityCRL"
      $SigninAssistantPresent=((Test-Path $SigninAssistantPath) -and ((Get-ItemProperty $SigninAssistantPath).TargetDir -ne $null))
      $MyCulture="0x{0:X}" -f(Get-Culture).LCID
      if (!$SigninAssistantPresent)
         {
         if ((Get-WmiObject Win32_OperatingSystem).OSArchitecture -eq "64 bits")
         	  # V7.250.4556.0
         	  {$Source="http://www.koxo.net/fichiers/msonline/msoidcli_64.msi"} ## {$Source="http://download.microsoft.com/download/A/7/5/A75FE163-6C03-4E44-8E81-70905727D8E6/msoidcli_64.msi"}
            else
            # V7.250.4556.0
            {$Source="http://www.koxo.net/fichiers/msonline/msoidcli_32.msi"} ## {$Source="http://download.microsoft.com/download/A/7/5/A75FE163-6C03-4E44-8E81-70905727D8E6/msoidcli_32.msi"}
         $Destination=$env:TEMP+"\_SigninAssistant_.msi"
         Write-Host "Trying to download ""Microsoft Online Sign-in Assistant"""
         Write-Host "Url=$($Source)"
         Write-Host "Please, wait ..." -ForegroundColor "Cyan"
         $wc.DownloadFile($Source,$Destination)
         $Install=Start-Process $Destination -Wait -PassThru
         if ($Install.ExitCode -eq 0)
            {
            $S="Microsoft Online Sign-in Assistant installation was OK"
            Write-Host $S -ForegroundColor "Green"
            Log_Message "OK" $S
            }
            else
            {
            $S="Installation failed for Microsoft Online Sign-in Assistant, code : $($Install.ExitCode)"
            Write-Host $S -ForegroundColor "Red"
            Log_Message "ERROR" $S
            } 
         $FileExists=Test-Path $Destination
         If ($FileExists) {Get-Item $Destination | Remove-Item} 
         }
      # MSOnline module
      $MSOnlinePath="HKLM:\SOFTWARE\Microsoft\MSOnlinePowerShell"
      $MSOnlinePresent=((Test-Path $MSOnlinePath) -and ((Get-ItemProperty $MSOnlinePath).InstallPath -ne $null))
      if (!$MSOnlinePresent)
         {
         if ((Get-WmiObject Win32_OperatingSystem).OSArchitecture -eq "64 bits")
            {
            # 64 bits
            if ([int](Get-WmiObject Win32_OperatingSystem).Version.Split(".")[0] -lt 6)
               {
               # Windows 2003
               $Source="http://www.koxo.net/fichiers/msonline/AdministrationConfig-fr-x64-patched.msi"
               }
               else
               {
               # Windows 2008 and above
               # V1.1.166.0
               $Source="http://www.koxo.net/fichiers/msonline/AdministrationConfig_3-x64.msi" ## $Source="http://download.connect.microsoft.com/pr/AdministrationConfig_3.msi?t=3edbe1ec-9076-441a-ba1d-6a8ae147d3ea&e=1497200079&h=81bba22aebadb1efceb708c41c45fa85"
               }
            }
            else
            {
            # 32 bits
            if ([int](Get-WmiObject Win32_OperatingSystem).Version.Split(".")[0] -lt 6)
               {
               # Windows 2003
               # V1.0
               $Source="http://www.koxo.net/fichiers/msonline/AdministrationConfig-fr-x86-patched.msi"
               }
               else
               {
               # Windows 2008 and above
               # V1.0
               $Source="http://www.koxo.net/fichiers/msonline/AdministrationConfig-fr-x86.msi" ## $Source="http://go.microsoft.com/fwlink/?linkid=236298" ## $Source=$Source+"&clcid="+$MyCulture
               }
            }
         $Destination=$env:TEMP+"\_MSOnline_.msi"
         write-host "Trying to download ""Microsoft Online Powershell Module"""
         Write-Host "Url=$($Source)"
         Write-Host "Please, wait ..." -ForegroundColor "Cyan"
         $wc.DownloadFile($Source,$Destination)
         $Install=Start-Process $Destination -Wait -PassThru
         if ($Install.ExitCode -eq 0)
            {
            $S="Microsoft Online Powershell Module installation was OK"
            Write-Host $S -ForegroundColor "Green"
            Log_Message "OK" $S
            }
            else
            {
            $S="Installation failed for Microsoft Online Powershell Module, code : $($Install.ExitCode)"
            Write-Host $S -ForegroundColor "Red"
            Log_Message "ERROR" $S
            }
         $FileExists=Test-Path $Destination
         if ($FileExists) {Get-Item $Destination | Remove-Item} 
         }
      # Import MSOnline Module
      $Mod=Import-Module "MSOnline" -PassThru
      if ([string]::IsNullOrEmpty($Mod)) 
         {
         $S="MSOnline Powershell module is still not present !"
         Write-Host $S -Foregroundcolor "Red"
         Log_Message "ERROR" $S
         Write-Host "Press any key ..."
         Read-Host
         }
         else
         {
         $S="MSOnline Powershell module is now present and loaded !"
         Write-Host $S -Foregroundcolor "Green"
         Log_Message "OK" $S
         $Res=$true
         }
      }
      else
      {
      Write-Host "MSOnline Powershell module is present" -Foregroundcolor "Green"
      $Res=$true
      }
  }
  else
  {
   $S="Please, install Framework .NET V3.5 first !"
   Write-Host $S -ForegroundColor "Red"
   Log_Message "ERROR" $S
   Write-Host "Press any key ..."
   Read-Host
  }
return $Res
}
###############################################################################
function connect_msol
{
Param($Cred)
Connect-MsolService -Credential $Cred	-ErrorAction SilentlyContinue
}
###############################################################################
function UserMSolExists
{
Param([string] $email)
$User=Get-MSolUser -UserPrincipalName $email -ErrorAction SilentlyContinue
return (![string]::IsNullOrEmpty($User))
}
###############################################################################
function LicensePackDomain
{
$sku=Get-MsolAccountSku
if ($sku.count -gt 0)
   {
   return $sku[0].AccountSkuId.Split(":")[0]
   }
   else
   {
   return ""
   }
}
###############################################################################
function user_info
{
Param([string] $identity)
$Error.Clear()
$User=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $identity {param($id) get-user -Identity $id}
return $User
}
###############################################################################
function usermailbox_info
{
Param([string] $identity)
$Error.Clear()
$User=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $identity {param($id) get-mailbox -Identity $id}
return $User
}
###############################################################################
function usermailbox_stat
{
Param([string] $identity)
$Error.Clear()
$User=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $identity {param($id) get-mailboxstatistics -Identity $id}
return $User
}
###############################################################################
function user_exists
{
Param([string] $identity)
$Error.Clear()
$User=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $identity {param($id) get-user -Identity $id}
return (![string]::IsNullOrEmpty($User))
}
###############################################################################
function group_exists
{
Param([string] $grp)
$Error.Clear()
$Group=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $grp {param($id) get-distributiongroup -Identity $id}
return (![string]::IsNullOrEmpty($Group))
}
###############################################################################
function group_info
{
Param([string] $grp)
$Error.Clear()
$Group=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $grp {param($id) get-group -Identity $id}
return $Group
}
###############################################################################
function distributiongroup_info
{
Param([string] $grp)
$Error.Clear()
$Group=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $grp {param($id) get-distributiongroup -Identity $id}
return $Group
}
###############################################################################
function distributiongroupmember_info
{
Param([string] $grp)
$Error.Clear()
$Group=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $grp {param($id) get-distributiongroupmember -Identity $id -ResultSize Unlimited}
$Group=$Group | Sort-Object -Property DisplayName
$Members=New-Object System.Collections.ArrayList
$Index=0
ForEach ($Member In $Group)
        {
        $Line="{0,-5}{1,-40} {2}" -f ($Index+1),$Member.DisplayName,$Member.WindowsLiveID
        $Index=$Members.Add($Line)
        }
return $Members
}
###############################################################################
function dynamicgroup_exists
{
Param([string] $grp)
$Error.Clear()
$Group=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $grp {param($id) get-dynamicdistributiongroup -Identity $id}
return (![string]::IsNullOrEmpty($Group))
}
###############################################################################
function dynamicgroup_info
{
Param([string] $grp)
$Error.Clear()
$Group=Invoke-Command -Session $Script:RS -ErrorAction SilentlyContinue -arg $grp {param($id) get-dynamicdistributiongroup -Identity $id}
return $Group
}
###############################################################################
function licenses_info
{
Param([string] $usr)
$Error.Clear()
$crlf = "`r`n"
if (UserMSolExists($usr))
   {
   $Prefix=LicensePackDomain
   $Licenses=(Get-MSolUser -UserPrincipalName $UserEmail -ErrorAction SilentlyContinue).Licenses
   if ([String]::IsNullorEmpty($error[0]))
      {
      ForEach ($License in $Licenses)
              {
              $Infos=$Infos+"Pack : "+$License.AccountSkuId+$crlf
              $ServicesStatus=$License.ServiceStatus
              ForEach ($ServiceStatus In  $ServicesStatus) 
                      {
                      $Infos=$Infos+"Plan : "+$ServiceStatus.ServicePlan.ServiceName+" ("+$ServiceStatus.ServicePlan.ServiceType+") ["+$ServiceStatus.ProvisioningStatus+"]"+$crlf
                      }
              }
      }
   } 
   else
   {
   $Infos="MSol User $($usr) not found !"+$crlf+$crlf
   }
return $Infos+$crlf
}
###############################################################################
function exec
{
Param([string] $ID,$GS,$GP,$EP)
$crlf = "`r`n"
if ( $ID -ne "%USER_"+"EMAIL%" )
   {
	if (user_exists($ID))
	{
        write-host "User : $($ID)"
		$Log.Text=$Log.Text+"------ Get-User ""$($ID)"" ------"
		$Info = user_info($ID) | fl | Out-String
		$Log.Text=$Log.Text+$Info
		$Log.Text=$Log.Text+"------ Get-Mailbox ""$($ID)"" ------"
		$Info = usermailbox_info($ID) | fl | Out-String
		$Log.Text=$Log.Text+$Info
		$Log.Text=$Log.Text+"------ Licenses ------"+$crlf+$crlf
        $Info=licenses_info($ID) 
		$Log.Text=$Log.Text+$Info
		$Log.Text=$Log.Text+"------ Get-MailboxStatistics ""$($ID)"" ------"
		$Info = usermailbox_stat($ID) | fl | Out-String
		$Log.Text=$Log.Text+$Info
	}
	else
	{
		$errors_count++
		$S = "User ""$($ID)"" does not exists"
		write-host $S -foregroundcolor "red"
	}
   }
if ( $GS -eq "%SECONDARY_"+"GROUP%" ) {$group = $GP} else {$group = $GS}
if ( $EP -ne "%SHARED_"+"SPACE%" ) {$group = $EP}
if (  ($ID -eq "%USER_"+"EMAIL%") -and (![string]::IsNullOrEmpty($group)) )
   {
	$grp = $false
	$grpd = $false
	if (dynamicgroup_exists($group))
	{
		write-host "Dynamic Group : $($group)"
		$Log.Text=$Log.Text+"------ Get-DynamicDistributionGroup ""$($group)"" ------"
		$Info = dynamicgroup_info($group) | fl | Out-String
		$Log.Text=$Log.Text+$Info
		$grpd = $true
	}
	if (group_exists($group))
	{
		write-host "Distribution Group : $($group)"
		$Log.Text=$Log.Text+"------ Get-Group ""$($group)"" ------"
		$GrpInfo = group_info($group)
		$Info = $GrpInfo | fl | Out-String
		$Members = $GrpInfo.Members |fl | Out-String
		$Log.Text=$Log.Text+$Info
		$Log.Text=$Log.Text+"------ Members ""$($group)"" ------"+$crlf
		$Log.Text=$Log.Text+$Members + $crlf + $crlf
		$Log.Text=$Log.Text+"------ Get-DistributionGroup ""$($group)"" ------"
		$Info = distributiongroup_info($group) | fl | Out-String
		$Log.Text=$Log.Text+$Info
		$Log.Text=$Log.Text+"------ Get-DistributionGroupMember ""$($group)"" ------"+$crlf+$crlf
		$Info = distributiongroupmember_info($group) | Out-String
		$Log.Text=$Log.Text+$Info
		$grp = $true
	}
	if (($grpd = $false) -and ($grp = $false)) 
	{
		$errors_count++
		$S = "Group ""$($group)"" does not exists"
		write-host $S -foregroundcolor "red"
	}
  }
}
###############################################################################
#                                                                             #
#                                    Main                                     #
#                                                                             #
###############################################################################
#
$global:Version="V3.97"
$global:total_errors_count=0
#
# Office 365 params
#
$Office365Url="https://ps.outlook.com/powershell/"
$Office365LoginName="%OFFICE365_ADMIN%"
$Office365LoginPassword=[System.Text.Encoding]::ASCII.GetString([System.Convert]::FromBase64String("#BASE64{%OFFICE365_PASSWORD%}"))
#
# Proxy parameters
#
$ProxyEnabled=$#BOOL{%PROXY_ENABLED%}
$ProxyUser="%PROXY_USER%"     
$ProxyPassword=[System.Text.Encoding]::ASCII.GetString([System.Convert]::FromBase64String("#BASE64{%PROXY_PASSWORD%}"))
#
$LogFile=$MyInvocation.MyCommand.Definition 
$LogFile=$LogFile -replace(".ps1",".log")
welcome
check_version
$Credential=connect $Office365Url $Office365LoginName $Office365LoginPassword $ProxyEnabled $ProxyUser $ProxyPassword
if (Install-MSOnlineCmdlets)
   {
   Connect-MSolService -Credential $Credential
   if ($global:total_errors_count -eq 0)
    {
    Connect-MSolService -Credential $Credential
    [void][Reflection.Assembly]::LoadWithPartialName("System.Windows.Forms")
    [void][Reflection.Assembly]::LoadWithPartialName("System.Drawing")
    $panel1 = new-object System.Windows.Forms.Panel
    $buttonClose = new-object System.Windows.Forms.Button
    $groupBox1 = new-object System.Windows.Forms.GroupBox
    $Log = new-object System.Windows.Forms.TextBox
    #
    # panel1
    #
    $panel1.Anchor = [System.Windows.Forms.AnchorStyles]"Bottom,Left,Right"
    $panel1.BackColor = [System.Drawing.Color]::FromArgb(30,139,225)
    $panel1.Controls.Add($buttonClose)
    $panel1.Location = new-object System.Drawing.Point(0, 402)
    $panel1.Name = "panel1"
    $panel1.Size = new-object System.Drawing.Size(673, 35)
    $panel1.TabIndex = 0
    #
    # buttonClose
    #
    $buttonClose.Anchor = [System.Windows.Forms.AnchorStyles]"Top,Bottom,Right"
    $buttonClose.Location = new-object System.Drawing.Point(579, 5)
    $buttonClose.Name = "buttonClose"
    $buttonClose.Size = new-object System.Drawing.Size(89, 25)
    $buttonClose.TabIndex = 0
    $buttonClose.Text = "&Close"
    $buttonClose.UseVisualStyleBackColor = $true
    function OnClick_buttonClose($Sender,$e){$MyForm.Close()}
    $buttonClose.Add_Click( { OnClick_buttonClose $buttonClose $EventArgs} )
    #
    # groupBox1
    #
    $groupBox1.Anchor = [System.Windows.Forms.AnchorStyles]"Top,Bottom,Left,Right"
    $groupBox1.BackColor = [System.Drawing.Color]::FromArgb(30,139,225)
    $groupBox1.Controls.Add($Log)
    $groupBox1.Location = new-object System.Drawing.Point(0, 0)
    $groupBox1.Name = "groupBox1"
    $groupBox1.Size = new-object System.Drawing.Size(672, 402)
    $groupBox1.TabIndex = 2
    $groupBox1.TabStop = $false
    $groupBox1.Text = " Informations "
    #
    # Log
    #
    $Log.AcceptsReturn = $true
    $Log.AcceptsTab = $true
    $Log.Anchor = [System.Windows.Forms.AnchorStyles]"Top,Bottom,Left,Right"
    $Log.BackColor = [System.Drawing.Color]::FromArgb(151,192,234)
    $Log.Cursor =[System.Windows.Forms.Cursors]::Default
    $Log.Font = new-object System.Drawing.Font("Courier New",10,[System.Drawing.FontStyle]::Bold)
    $Log.Location = new-object System.Drawing.Point(6, 16)
    $Log.Multiline = $true
    $Log.Name = "Log"
    $Log.ScrollBars =[System.Windows.Forms.ScrollBars]::Both
    $Log.Size = new-object System.Drawing.Size(660, 382)
    $Log.TabIndex = 0
    $Log.WordWrap = $false
    #
    $MyForm = new-object System.Windows.Forms.form
    #
    $MyForm.AutoSize = $true
    $MyForm.ClientSize = new-object System.Drawing.Size(672, 440)
    $MyForm.Controls.Add($groupBox1)
    $MyForm.Controls.Add($panel1)
    $MyForm.Name = "MyForm"
    $MyForm.StartPosition =[System.Windows.Forms.FormStartPosition]::CenterScreen
    $MyForm.Text = "(C)2006-2019 KoXo Developpement | Office 365 Powershell Log"
    #
    ### BEGIN_REPETITION ###
    $UserEmail="%USER_EMAIL%"
    $SecondaryGroup="#LOWER{#REMOVE_SPACE{%SECONDARY_GROUP%}}"
    $PrimaryGroup="#LOWER{#REMOVE_SPACE{%PRIMARY_GROUP%}}"
    $SharedSpace="#LOWER{#REMOVE_SPACE{%SHARED_SPACE%}}"
    exec $UserEmail $SecondaryGroup $PrimaryGroup $SharedSpace
    ### END_REPETITION ###
    #
    $MyForm.Topmost = $False
    $MyForm.Add_Shown({$MyForm.Activate()})
    [void] $MyForm.ShowDialog()
    $MyForm.Dispose()
    # close
    }
   } 
